/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/


/* ---------------------------------------------------------------
 * File: xpci32.c 
 * -----------------------------------------------------------------*/

#include <stdio.h>
#include <windows.h>
#include <winioctl.h>
#include <b_ioctl.h>

#include <xpci.h>
#include <xpci32rg.h>
#include <xsession.h>
#include <xlasterr.h>
#include <timeout.h>

#include <assert.h>

extern bx_int32 GlobalBaseAddr;


/* xpci.h defines a bitfield which encapsulates the following structure
   the bitfield is not used in this file but other source files rely on it */

/* slotnumber and devid format:   bit 15..8  - bus number,
                                  bit  7..3  - unit number (slot),
                                  bit  2..0  - function
*/


/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXDirectRegPCIRead(
 *
 * Purpose: read register using access port
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXDirectRegPCIRead(
  bx_handletype handle,
  bx_int32 dir_addr,
  bx_int32 regsize,
  bx_int32 *pValue
  )
{
  HANDLE osHandle = bx_handlearray[handle].portnumber;
  BOOL fIoCtlSuccess;
  b_accessporttype accessPort;
  ULONG outBuffer=0, readBytes=0;

  assert (("NULL passed to BestXDirectRegPCIRead", pValue));

  if (osHandle == INVALID_OS_HANDLE)
  {
    return BX_E_NOT_CONNECTED;
  }

  accessPort.m_dwAddress = dir_addr;
  accessPort.m_cRegWidth = (UCHAR) (regsize & 0xFF);

  fIoCtlSuccess = DeviceIoControl (
    osHandle,                       /* os Handle */
    (ULONG) IOCTL_NTIO_READ_REG_DW, /* command */
    &accessPort,                    /* data to driver */
    sizeof(accessPort),             /* length of data to driver */
    &outBuffer,                     /* data from driver*/
    sizeof (outBuffer),             /* length of data from driver */
    &readBytes,                     /* actually read bytes in buf */
    NULL                            /* wait for completion */
    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

  if (!fIoCtlSuccess) {
    /* TODO: find another error message, CZ */
    return BX_E_ERROR;
  }
  
  *pValue = outBuffer;

  return BX_E_OK;
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXDirectRegPCIWrite(
 *
 * Purpose: write register using access port
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXDirectRegPCIWrite(
  bx_handletype handle,
  bx_int32 dir_addr,
  bx_int32 regsize,    
  bx_int32 reg_value
  )
{
  HANDLE osHandle = bx_handlearray[handle].portnumber;
  BOOL fIoCtlSuccess;
  b_accessportdatatype accessPortData;
  ULONG writtenBytes=0;

  if (osHandle == INVALID_OS_HANDLE)
  {
    return BX_E_NOT_CONNECTED;
  }

  accessPortData.m_port.m_dwAddress = dir_addr;
  accessPortData.m_port.m_cRegWidth = (UCHAR) (regsize & 0xFF);
  accessPortData.m_data = reg_value;

  fIoCtlSuccess = DeviceIoControl (
    osHandle,                        /* os Handle */
    (ULONG) IOCTL_NTIO_WRITE_REG_DW, /* command */
    &accessPortData,                 /* data to driver */
    sizeof(accessPortData),          /* length of data to driver */
    NULL,                            /* data from driver*/
    0,                               /* length of data from driver */
    &writtenBytes,                   /* actually written bytes in buf */
    NULL                             /* wait for completion */
    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

  if (!fIoCtlSuccess) {
    /* TODO: find another error message, CZ */
    return BX_E_ERROR;
  }
  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * STATICS
 * -------------------------------------------------------------------------- */

bx_errtype EXPORT BestXPCIDWGet(
    bx_int32 Reg_N,              /* offset into config. space */
    bx_portnumtype OsHandle,
    bx_int32 * pResult);

bx_errtype EXPORT BestXPCIDWSet(
    bx_int32 Reg_N,              /* offset into config. space */
    bx_portnumtype OsHandle,
    bx_int32 value);

/* --------------------------------------------------------------------------
 * Open port, read from a config register, close port
 * -------------------------------------------------------------------------- */

bx_errtype EXPORT BestXGenericPCIDWGet(bx_int32 dwHPSlotId,
    bx_int32 Reg_N,
    bx_int32 * pResult)
{
  BOOL fIoCtlSuccess;
  ULONG ReturnedLength=0;
  bx_int32 DataBuffer=0;
  HANDLE OsHandle;

  bx_errtype lErrCode;
  assert(pResult && "Null passed to BestXGenericPCIDWGet()");

  /* SCR; 23/5/97; changed return type of BestXOpenPCI() */
  if (BX_E_OK != (lErrCode = BestXOpenPCI((int) dwHPSlotId, &OsHandle)))
    return lErrCode;

  if (INVALID_OS_HANDLE == OsHandle)
    return BX_E_PCI_OPEN;

  fIoCtlSuccess = DeviceIoControl(
    OsHandle,                       /* Handle to device  */
    (ULONG) IOCTL_NTIO_GET_CONFIG_DW, /* IO Control code */
    &Reg_N,                     /* Buffer to driver. */
    sizeof(Reg_N),              /* Length of buffer in bytes. */
    &DataBuffer,                /* Buffer from driver.  */
    sizeof(DataBuffer),         /* Length of buffer in bytes. */
    &ReturnedLength,            /* Bytes placed in DataBuffer. */
    NULL                        /* NULL means wait till op. completes. */
    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

  (void) BestXClosePCI((bx_portnumtype) OsHandle);

  if (!fIoCtlSuccess) {
    /* TODO: find another error message, CZ */
    return BX_E_ERROR;
  }
  
  *pResult = DataBuffer;
  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * Open port, write to a config register, close port
 * -------------------------------------------------------------------------- */

bx_errtype EXPORT BestXGenericPCIDWSet(bx_int32 dwHPSlotId,
    bx_int32 Reg_N,
    bx_int32 value)
{
  BOOL fIoCtlSuccess;
  ULONG ReturnedLength=0;
  b_configdrivertype WriteBuffer;
  HANDLE OsHandle;
  bx_errtype lErrCode;

  WriteBuffer.Offset = Reg_N;
  WriteBuffer.Data.Data = value;

  /* SCR; 23/5/97; changed return type of BestXOpenPCI() */
  if (BX_E_OK != (lErrCode = BestXOpenPCI((int) dwHPSlotId, &OsHandle)))
    return lErrCode;

  if (INVALID_OS_HANDLE == OsHandle)
    return BX_E_PCI_OPEN;

  fIoCtlSuccess = DeviceIoControl(
    OsHandle,                       /* Handle to device  */
    (ULONG) IOCTL_NTIO_SET_CONFIG_DW, /* IO Control code */
    &WriteBuffer,               /* Buffer to driver. */
    sizeof(WriteBuffer),        /* Length of buffer in bytes. */
    NULL,                       /* Buffer from driver.  */
    0,                          /* Length of buffer in bytes. */
    &ReturnedLength,            /* Bytes placed in DataBuffer. */
    NULL                        /* NULL means wait till op. completes. */
    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

  (void) BestXClosePCI((bx_portnumtype) OsHandle);

  if (!fIoCtlSuccess)
    return BX_E_PCI_OPEN;

  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * Reads from a config register
 * -------------------------------------------------------------------------- */

static bx_errtype BestXPCIDWGet(
    bx_int32 Reg_N,              /* offset into config. space */
    bx_portnumtype OsHandle,
    bx_int32 * pResult)
{
  BOOL fIoCtlSuccess;
  ULONG ReturnedLength=0;
  bx_int32 DataBuffer=0;
  assert(pResult && "Null passed to BestXPCIDWGet()");

  fIoCtlSuccess = DeviceIoControl(
    (HANDLE) OsHandle,          /* Handle to device  */
    (ULONG) IOCTL_NTIO_GET_CONFIG_DW, /* IO Control code for Read */
    &Reg_N,                     /* Buffer to driver. */
    sizeof(Reg_N),              /* Length of buffer in bytes. */
    &DataBuffer,                /* Buffer from driver.  */
    sizeof(DataBuffer),         /* Length of buffer in bytes. */
    &ReturnedLength,            /* Bytes placed in DataBuffer. */
    NULL                        /* NULL means wait till op. completes. */
    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

  if (!fIoCtlSuccess) {
    /* TODO: this error requires a parameter and therefore a handle */
    return BX_E_ERROR;
  }
  
  *pResult = DataBuffer;
  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * Writes to a config register
 * -------------------------------------------------------------------------- */

static bx_errtype BestXPCIDWSet(
    bx_int32 Reg_N,
    bx_portnumtype OsHandle,
    bx_int32 value)
{
  BOOL fIoCtlSuccess;
  ULONG ReturnedLength=0;
  b_configdrivertype WriteBuffer;
  WriteBuffer.Offset = Reg_N;
  WriteBuffer.Data.Data = value;

  fIoCtlSuccess = DeviceIoControl(
    (HANDLE) OsHandle,          /* Handle to device  */
    (ULONG) IOCTL_NTIO_SET_CONFIG_DW, /* IO Control code for Read */
    &WriteBuffer,               /* Buffer to driver. */
    sizeof(WriteBuffer),        /* Length of buffer in bytes. */
    NULL,                       /* Buffer from driver.  */
    0,                          /* Length of buffer in bytes. */
    &ReturnedLength,            /* Bytes placed in DataBuffer. */
    NULL                        /* NULL means wait till op. completes. */
    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

  if (!fIoCtlSuccess) {
    /* TODO: this error requires a parameter and therefore a handle, CZ */
    return BX_E_ERROR;
  }
  
  return BX_E_OK;
}

/* ----------------------------------------------------------------------------
 * ISP using MemRd/MemWr. Get Value
 * ------------------------------------------------------------------------- */
bx_errtype EXPORT BestXGetISPDWMem
(
  bx_int32 Reg_N,
  bx_portnumtype OsHandle,
  bx_int32 * pResult
)
{
  BOOL fIoCtlSuccess;
  ULONG ReturnedLength = 0;
  bx_int32 DataBuffer = 0;

  assert( pResult && "Null passed to BestXGetISPDWMem()" );

  fIoCtlSuccess = DeviceIoControl(
    (HANDLE) OsHandle,                  /* Handle to device  */
    (ULONG) IOCTL_NTIO_GET_ISP_DW_MEM,  /* IO Control code for Read */
    &Reg_N,                             /* Buffer to driver. */
    sizeof(Reg_N),                      /* Length of buffer in bytes. */
    &DataBuffer,                        /* Buffer from driver.  */
    sizeof(DataBuffer),                 /* Length of buffer in bytes. */
    &ReturnedLength,                    /* Bytes placed in DataBuffer. */
    NULL                                /* NULL means wait till op. completes. */
    );

  DBG_ApiFailureIsZero( fIoCtlSuccess );

  if( !fIoCtlSuccess )
  {
    /* TODO: this error requires a parameter and therefore a handle */
    return BX_E_ERROR;
  }
  
  *pResult = DataBuffer;

  return BX_E_OK;
}

/* ----------------------------------------------------------------------------
 * ISP using MemRd/MemWr. Set Value
 * ------------------------------------------------------------------------- */
bx_errtype EXPORT BestXSetISPDWMem
(
  bx_int32 Reg_N,
  bx_portnumtype OsHandle,
  bx_int32 value
)
{
  BOOL fIoCtlSuccess;
  ULONG ReturnedLength = 0;
  b_configdrivertype WriteBuffer;
  WriteBuffer.Offset = Reg_N;
  WriteBuffer.Data.Data = value;

  fIoCtlSuccess = DeviceIoControl(
    ( HANDLE )OsHandle,                 /* Handle to device  */
    ( ULONG )IOCTL_NTIO_SET_ISP_DW_MEM, /* IO Control code for Read */
    &WriteBuffer,                       /* Buffer to driver. */
    sizeof(WriteBuffer),                /* Length of buffer in bytes. */
    NULL,                               /* Buffer from driver.  */
    0,                                  /* Length of buffer in bytes. */
    &ReturnedLength,                    /* Bytes placed in DataBuffer. */
    NULL                                /* NULL means wait till op. completes. */
    );

  DBG_ApiFailureIsZero( fIoCtlSuccess );

  if( !fIoCtlSuccess )
  {
    /* TODO: this error requires a parameter and therefore a handle, CZ */
    return BX_E_ERROR;
  }
  
  return BX_E_OK;
}

/* --------------------------------------------------------------------------
 * Should only be called from BestXSetRegwidth()
 * -------------------------------------------------------------------------- */

bx_errtype BestXPCISetRegwidth(bx_portnumtype OsHandle, int iRegWidth)
{
  BOOL fIoCtlSuccess;
  ULONG ulReturnedLength=0;
  USHORT usRegWidth = (USHORT)iRegWidth;  /* driver needs USHORT */

  fIoCtlSuccess = DeviceIoControl(
    OsHandle,                   /* Handle to device */
    (ULONG) IOCTL_NTIO_SET_REGWIDTH_PCI,  /* IO Control code for Write */
    &usRegWidth,                /* Buffer to driver.  Holds port & data. */
    sizeof(usRegWidth),         /* Length of buffer in bytes. */
    NULL,                       /* Buffer from driver.   Not used. */
    0,                          /* Length of buffer in bytes.  */
    &ulReturnedLength,          /* Bytes placed in outbuf.  Should be 0. */
    NULL                        /* NULL means wait till I/O completes. */
    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

  if (!fIoCtlSuccess) {
    /* TODO: this error requires a param */
    return BX_E_ERROR;
  }
  
  return BX_E_OK;
}

/* --------------------------------------------------------------------------
 * Should only be called from BestXDeviceConnect()
 * -------------------------------------------------------------------------- */

bx_errtype BestXPCIDeviceConnect(bx_portnumtype OsHandle)
{
  /* try to connect */
  if (BestXPCIDWSet(PCI_CONNECT_CMD, OsHandle, PCI_CONNECT_CMD_BIT) != BX_E_OK) 
  {
    /* TODO: error param */
    return BX_E_ERROR;
  }
  
  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * Should only be called from BestXCheckConnection()
 * -------------------------------------------------------------------------- */

bx_errtype BestXPCICheckConnection(bx_portnumtype OsHandle)
{
  bx_errtype lErrCode;
  bx_int32 Result;

  if (BX_E_OK != (lErrCode = BestXPCIDWGet(PCI_CONNECT_STATUS, OsHandle, &Result)))
    return lErrCode;

  if ((Result & PCI_CONNECT_STATUS_BIT) == PCI_CONNECTION_CLOSED)
    return BX_E_NOT_CONNECTED;

  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * Should only be called from BestXReleaseConnection()
 * -------------------------------------------------------------------------- */

void BestXPCIReleaseConnection(bx_portnumtype OsHandle)
{
  (void) BestXPCIDWSet(PCI_CONNECT_CMD, OsHandle, 0x0);
  return;
}

#ifndef DOSW32
/* --------------------------------------------------------------------------
 * Should only be called from BestXPortTimeoutSet()
 * -------------------------------------------------------------------------- */
bx_errtype BestXPciPortTimeoutSet(bx_portnumtype OsHandle, 
                                BESTTIMEOUTS * pCallersTimeouts)
{
  BOOL fIoCtlSuccess;
  ULONG ReturnedLength = 0;

  fIoCtlSuccess = DeviceIoControl(
    (HANDLE) OsHandle,          /* Handle to device  */
    (ULONG) IOCTL_NTIO_SET_TIMEOUT, /* IO Control code for setting timeouts */
    (PVOID)pCallersTimeouts,    /* Buffer to driver. */
    sizeof(BESTTIMEOUTS),       /* Length of buffer in bytes. */
    NULL,                       /* Buffer from driver.  */
    0,                          /* Length of buffer in bytes. */
    &ReturnedLength,            /* Bytes placed in DataBuffer. */
    NULL                        /* NULL means wait till op. completes. */
    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

  if (!fIoCtlSuccess) {
    /* TODO: error param */
    return BX_E_ERROR;
  }
  
  return BX_E_OK;
}

#endif

/* --------------------------------------------------------------------------
 * Because not all single-function devices correctly respond to a read
 * of functions 1 through 7 (even though they're not implemented) this
 * function allows a direct way of determining the type of device.
 * -------------------------------------------------------------------------- */

bx_errtype BestXPciDevMultiFuncCheck(bx_int32 dwHPSlotId, bx_bool * fIsMultiFunc)
{
  bx_int32 Result;

  *fIsMultiFunc = 0;

  if ( BX_E_OK != BestXGenericPCIDWGet(dwHPSlotId, PCI_MF_OFFSET, &Result))
    return BX_E_NO_BEST_PCI_DEVICE_FOUND;

  *fIsMultiFunc = ((Result & PCI_MF_BIT) != 0);
  return BX_E_OK;
}


#ifndef DOSW32

/*
////////////////////////////////////////////////////////////////////////////////////
// SCR; 5/22/97; radical changes when using the registry (multiple devices)
//
*/
bx_errtype EXPORT BestXOpenPCI(int intHPSlotId, bx_portnumtype * pOsHandle)
{

#define     MAX_CREATE_NAME_LEN     30

  HANDLE OsHandle;

  bx_int32 dwHPSlotId;
  char cDeviceName[MAX_CREATE_NAME_LEN];
  bx_int32 dwBuffLen = MAX_CREATE_NAME_LEN;
  bx_int32 dwCurrOwnerProcessId;
  bx_errtype lErrCode;
  assert(pOsHandle && "Null passed to BestXOpenPCI()");

  /* init for failure */
  *pOsHandle = INVALID_OS_HANDLE;

  /* check the version here */
  /* *** BX_E_OK is returned if _DEBUG is defined *** */
  /* *** ALWAYS TEST the Release version *** */
  if (BX_E_OK != (lErrCode = BestXNTRegVersionCheck()))
    return lErrCode;

  /* transfer to proper variable */
  dwHPSlotId = (bx_int32) intHPSlotId;

  /* get the name that we'll use to open the device */
  if (BX_E_OK != (lErrCode = BestXNTRegDevNameGet(dwHPSlotId, cDeviceName, &dwBuffLen)))
    return lErrCode;

  /* now try to open the device */
  OsHandle = CreateFile(cDeviceName,
    GENERIC_WRITE | GENERIC_READ,
    FILE_SHARE_WRITE,
    NULL,
    OPEN_EXISTING,
    0,
    NULL);

  if (INVALID_HANDLE_VALUE == OsHandle)
  {
    DBG_ApiLastError;
    return BX_E_PCI_OPEN;
  }

  /* register ourselves as the device-owner */
  /* **TODO** figure out what we want to do if this fails ??? */
  lErrCode = BestXNTRegOwnerSet(dwHPSlotId, &dwCurrOwnerProcessId);

  *pOsHandle = (bx_portnumtype) OsHandle;

  return BX_E_OK;
}


/*
////////////////////////////////////////////////////////////////////////////////////
// SCR; 5/27/97; The only difference between Direct I/O and PCI access is
//               reflected in the open call.
//
*/
bx_errtype BestXOpenIO(int intHPSlotId, bx_portnumtype * pOsHandle)
{
  BOOL fIoCtlSuccess;
  ULONG ReturnedLength=0;
  bx_int32 WriteBuffer = 0;

  bx_errtype lErrCode;
  assert(pOsHandle && "Null passed to BestXOpenIO()");

  /* open the standard pci driver  */
  if (BX_E_OK != (lErrCode = BestXOpenPCI(intHPSlotId, pOsHandle)))
    return lErrCode;

  /* set Direct I/O mode */
  fIoCtlSuccess = DeviceIoControl(
    *pOsHandle,                 /* Handle to device   */
    (ULONG) IOCTL_NTIO_SET_DIRECT_IO, /* IO Control code */
    NULL,                       /* Buffer to driver.  */
    0,                          /* Length of buffer in bytes.  */
    NULL,                       /* Buffer from driver.   */
    0,                          /* Length of buffer in bytes.  */
    &ReturnedLength,            /* Bytes placed in DataBuffer.  */
    NULL                        /* NULL means wait till op. completes.  */
    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

  if (fIoCtlSuccess)
  {
    /* and get BaseAddress (future music) */
    fIoCtlSuccess = DeviceIoControl(
      *pOsHandle,               /* Handle to device   */
      (ULONG) IOCTL_NTIO_GET_BASE_ADDR, /* IO Control code */
      NULL,                     /* Buffer to driver.  */
      0,                        /* Length of buffer in bytes.  */
      &WriteBuffer,             /* Buffer from driver.   */
      sizeof(ULONG),            /* Length of buffer in bytes.  */
      &ReturnedLength,          /* Bytes placed in DataBuffer.  */
      NULL                      /* NULL means wait till op. completes.  */
      );

    DBG_ApiFailureIsZero(fIoCtlSuccess);

  }

  if (!fIoCtlSuccess || (sizeof(ULONG) != ReturnedLength))
  {
    (void) CloseHandle(*pOsHandle);
    *pOsHandle = INVALID_OS_HANDLE;
    GlobalBaseAddr = 0xffffffff;
    return BX_E_PROG_DEC_ENABLE;
  }

  GlobalBaseAddr = WriteBuffer;
  return BX_E_OK;
}

#endif /*DOSW32*/

/*
/////////////////////////////////////////////////////////////////////////////
// Returns the HPSlotId from the Handle array at the open element
// containing the OS file handle "OsHandle"
*/
bx_errtype BestXGetHPSlotIdFromOsHandle(
    bx_portnumtype OsHandle,
    bx_int32 * pdwHPSlotId
)
{
  bx_handletype index;
  assert(pdwHPSlotId && "Null passed to BestXGetHPSlotIdFromOsHandle()");

  for (index = 0; index < BX_MAXHANDLES; index++)
  {
    if (bx_handlearray[index].is_open &&
      bx_handlearray[index].portnumber == OsHandle
      )
    {
      *pdwHPSlotId = bx_handlearray[index].entered_port;
      return BX_E_OK;
    }
  }

  return BX_E_BAD_HANDLE;
}

/*
//////////////////////////////////////////////////////////////////////////////
// SCR; 5/22/97; radical changes when using the registry (multiple devices)
*/
bx_errtype EXPORT BestXClosePCI(bx_portnumtype OsHandle)
{
  bx_int32 dwHPSlotId;
  bx_int32 dwCurrOwnerProcessId;
  /* get the slot id before the OS call  */
  /* (before anybody can mess with the handle) */
  /* we're going to ignore errors here...deliberately */
  (void) BestXGetHPSlotIdFromOsHandle(OsHandle, &dwHPSlotId);

  /* OS call to close device */
  if (!CloseHandle((HANDLE) OsHandle))
  {
    return BX_E_BAD_HANDLE;
  }

  /* release ownership of the device */
  /* **TODO** figure out what we want to do if this fails ??? */
  (void) BestXNTRegOwnerClear(dwHPSlotId, &dwCurrOwnerProcessId);

  OsHandle = NULL;
  return BX_E_OK;
}

/*
//////////////////////////////////////////////////////////////////////////////
*/
bx_errtype BestXPCIMailboxWrite(bx_int32 dwHPSlotId, bx_int32 value)
{
  bx_int32 statusdata;
  BEST_TIMER tmr;
  bx_errtype err;

  /* init timeout */
  BestStartTimeout(TIMEOUT_MAILBOX_PCI_WRITE, &tmr);

  do
  {
    /* get Mailbox status */
    if (BX_E_OK != (err = BestXGenericPCIDWGet(dwHPSlotId, PCI_MBOX_STATUS, &statusdata)))
      return err;

    if ( BestIsTimeoutDone(&tmr) ) {
      /* TODO: error param */
      return BX_E_MAILBOX_TIMEOUT;  /* a timeout here is an error */
    }
    
  } while ((statusdata & PCI_MBOX_SEND_FULL_BIT) != 0);

  /* write data to Mailbox */
  return BestXGenericPCIDWSet(dwHPSlotId, PCI_MBOX_DATA, value);
}


/*
//////////////////////////////////////////////////////////////////////////////
*/
bx_errtype BestXPCIMailboxRead(bx_int32 dwHPSlotId, bx_int32 * pResult)
{
  bx_int32 statusdata;
  BEST_TIMER  tmr;
  bx_errtype err;
  assert(pResult && "Null passed to BestXPCIMailboxRead()");

  /* init timeout */
  BestStartTimeout(TIMEOUT_MAILBOX_PCI_READ, &tmr);

  do
  {
    /* get status from Config */
    if (BX_E_OK != (err = BestXGenericPCIDWGet(dwHPSlotId, PCI_MBOX_STATUS, &statusdata)))
      return err;

    if ( BestIsTimeoutDone(&tmr) ) {
      /* TODO: error param */
      return BX_E_MAILBOX_TIMEOUT;               /* a timeout here is an error */
    }
    
  } while ((statusdata & PCI_MBOX_DATA_VALID_BIT) == 0);

  /* get data from data register */
  if (BX_E_OK != (err = BestXGenericPCIDWGet(dwHPSlotId, PCI_MBOX_DATA, &statusdata)))
    return err;

  /* get data from data register */
  *pResult = statusdata;

  /* clear the data valid bit */
  return BestXGenericPCIDWSet(dwHPSlotId, PCI_MBOX_STATUS, PCI_MBOX_DATA_VALID_BIT);
}


